/****************************************************************************
 *
 *   BSD LICENSE
 * 
 *   Copyright(c) 2007-2023 Intel Corporation. All rights reserved.
 *   All rights reserved.
 * 
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 * 
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Intel Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 * 
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 *  version: QAT20.L.1.2.30-00078
 *
 ***************************************************************************/

/**
 *****************************************************************************
 * @file dc_header_footer_lz4.h
 *
 * @ingroup Dc_DataCompression
 *
 * @description
 *      Definition of the Data Compression header and footer parameters
 *      for CPM 2.0.
 *
 *****************************************************************************/
#ifndef DC_HEADER_FOOTER_LZ4_H_
#define DC_HEADER_FOOTER_LZ4_H_

#include "lac_common.h"

/* Header and footer size LZ4 */
#define DC_LZ4_HEADER_SIZE 7
#define DC_LZ4_FOOTER_SIZE 8

/* Values used to build the headers for LZ4 */
#define DC_LZ4_FH_ID 0x184D2204U
#define DC_LZ4_FH_FLG_VERSION 0x1
#define DC_LZ4_FH_MAX_BLK_SIZE_ENUM_MIN 4

/* Values used to build footers for LZ4 */
#define DC_LZ4_FF_END_MARK 0x0

typedef enum _CpaDcFrameType
{
    CPA_DC_FRAME_ZLIB,
    CPA_DC_FRAME_GZIP,
    CPA_DC_FRAME_LZ4
} CpaDcFrameType;

/**
 *****************************************************************************
 * @ingroup dc_lz4_generate_header
 *      Generate the LZ4 Header.
 *
 * @description
 *      This function generates the LZ4 compression header.
 *
 * @param[in]       dest_buff        Pointer to the destination buffer the
 *                                   LZ4 header will be written to.
 * @param[in]       max_block_size   LZ4 Maximum block size.
 * @param[in]       block_indep       LZ4 block independence value.
 * @param[in,out]   count            Pointer to counter that stores
 *                                   amount of generated bytes.
 *
 * @retval CPA_STATUS_SUCCESS        Function executed successfully.
 * @retval CPA_STATUS_INVALID_PARAM  Invalid parameter passed in.
 *****************************************************************************/
CpaStatus dc_lz4_generate_header(const CpaFlatBuffer *dest_buff,
                                 const CpaDcCompLZ4BlockMaxSize max_block_size,
                                 const CpaBoolean block_indep,
                                 Cpa32U *count);

/**
 *****************************************************************************
 * @ingroup dc_lz4_generate_footer
 *      Generate the LZ4 footer.
 *
 * @description
 *      This function generates tha LZ4 compression footer.
 *
 * @param[in]   dest_buff        Pointer to the destination buffer where
 *                               the LZ4 footer will be written to.
 * @param[in]   pRes             Pointer to the CpaDcRqResults structure
 *                               that holds XXHASH32.
 *
 * @retval CPA_STATUS_SUCCESS        Function executed successfully.
 * @retval CPA_STATUS_INVALID_PARAM  Invalid parameter passed in.
 *****************************************************************************/
CpaStatus dc_lz4_generate_footer(const CpaFlatBuffer *dest_buff,
                                 const CpaDcRqResults *pRes);

#endif /* DC_HEADER_FOOTER_LZ4_H_ */
