/***************************************************************************
 *
 *   BSD LICENSE
 * 
 *   Copyright(c) 2007-2023 Intel Corporation. All rights reserved.
 *   All rights reserved.
 * 
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 * 
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in
 *       the documentation and/or other materials provided with the
 *       distribution.
 *     * Neither the name of Intel Corporation nor the names of its
 *       contributors may be used to endorse or promote products derived
 *       from this software without specific prior written permission.
 * 
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 *  version: QAT20.L.1.2.30-00078
 *
 ***************************************************************************/

/**
 *****************************************************************************
 * @file lac_sym_qat_hash_defs_lookup.h
 *
 * @defgroup LacSymQatHashDefsLookup  Hash Defs Lookup
 *
 * @ingroup  LacSymQatHash
 *
 * API to be used for the hash defs lookup table.
 *
 *****************************************************************************/

#ifndef LAC_SYM_QAT_HASH_DEFS_LOOKUP_P_H
#define LAC_SYM_QAT_HASH_DEFS_LOOKUP_P_H

#include "cpa.h"
#include "cpa_cy_sym.h"

/**
******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      Finishing Hash algorithm
* @description
*      This define points to the last available hash algorithm
* @NOTE: If a new algorithm is added to the api, this #define
* MUST be updated to being the last hash algorithm in the struct
* CpaCySymHashAlgorithm in the file cpa_cy_sym.h
*****************************************************************************/
#define CPA_CY_HASH_ALG_END CPA_CY_SYM_HASH_SM3

/**
******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      Default authKeyLenInBytes
* @description
*      This define the default value for authKeyLenInBytes used in
*      LacSymQat_HashAlgLookupGet and LacSymQat_HashDefsLookupGet
*****************************************************************************/
#define LAC_SYM_HASH_DEFAULT_AUTHKEY_LENS 0

/**
******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      Default digestResultLenInBytes
* @description
*      This define the default value for digestResultLenInBytes used in
*      LacSymQat_HashAlgLookupGet and LacSymQat_HashDefsLookupGet
*****************************************************************************/
#define LAC_SYM_HASH_DEFAULT_DIGEST_LENS 0

/***************************************************************************/

/**
******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      hash algorithm specific structure
* @description
*      This structure contain constants specific to an algorithm.
*****************************************************************************/
typedef struct lac_sym_qat_hash_alg_info_s
{
    Cpa32U digestLength; /**< Digest length in bytes */
    Cpa32U blockLength;  /**< Block length in bytes */
    Cpa8U *initState;    /**< Initialiser state for hash algorithm */
    Cpa32U stateSize;    /**< size of above state in bytes */
} lac_sym_qat_hash_alg_info_t;

/**
******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      hash qat specific structure
* @description
*      This structure contain constants as defined by the QAT for an
*      algorithm.
*****************************************************************************/
typedef struct lac_sym_qat_hash_qat_info_s
{
    Cpa32U algoEnc;      /**< QAT Algorithm encoding */
    Cpa32U authCounter;  /**< Counter value for Auth */
    Cpa32U state1Length; /**< QAT state1 length in bytes */
    Cpa32U state2Length; /**< QAT state2 length in bytes */
} lac_sym_qat_hash_qat_info_t;

/**
******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      hash defs structure
* @description
*      This type contains pointers to the hash algorithm structure and
*      to the hash qat specific structure
*****************************************************************************/
typedef struct lac_sym_qat_hash_defs_s
{
    lac_sym_qat_hash_alg_info_t *algInfo;
    /**< pointer to hash info structure */
    lac_sym_qat_hash_qat_info_t *qatInfo;
    /**< pointer to hash QAT info structure */
} lac_sym_qat_hash_defs_t;

/**
*******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      initialise the hash lookup table
*
* @description
*      This function initialises the digest lookup table.
*
* @note
*      This function does not have a corresponding shutdown function.
*
* @return CPA_STATUS_SUCCESS   Operation successful
* @return CPA_STATUS_RESOURCE  Allocating of hash lookup table failed
*
*****************************************************************************/
CpaStatus LacSymQat_HashLookupInit(CpaInstanceHandle instanceHandle);

/**
*******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      get hash algorithm specific structure from lookup table
*
* @description
*      This function looks up the hash lookup array for a structure
*      containing data specific to a hash algorithm. The hashAlgorithm enum
*      value MUST be in the correct range prior to calling this function.
*
* @param[in]  hashAlgorithm          Hash Algorithm
* @param[in]  authKeyLenInBytes      Length of the authentication key in bytes.
*                                    only used for ZUC-EIA/AES_CMAC algorithms.
*                                    default value 0 for the other algorithms.
* @param[in]  digestResultLenInBytes Length of the hash result in bytes
*                                    only used for ZUC-EIA/AES_CMAC algorithms.
*                                    default value 0 for the other algorithms.
* @param[out] ppHashAlgInfo          Hash Alg Info structure
*
* @return None
*
*****************************************************************************/
void LacSymQat_HashAlgLookupGet(CpaInstanceHandle instanceHandle,
                                CpaCySymHashAlgorithm hashAlgorithm,
                                lac_sym_qat_hash_alg_info_t **ppHashAlgInfo,
                                Cpa32U authKeyLenInBytes,
                                Cpa32U digestResultLenInBytes);

/**
*******************************************************************************
* @ingroup LacSymQatHashDefsLookup
*      get hash defintions from lookup table.
*
* @description
*      This function looks up the hash lookup array for a structure
*      containing data specific to a hash algorithm. This includes both
*      algorithm specific info and qat specific infro. The hashAlgorithm enum
*      value MUST be in the correct range prior to calling this function.
*
* @param[in]  hashAlgorithm          Hash Algorithm
* @param[in]  authKeyLenInBytes      Length of the authentication key in bytes.
*                                    only used for ZUC-EIA/AES_CMAC algorithms.
*                                    default value 0 for the other algorithms.
* @param[in]  digestResultLenInBytes Length of the hash result in bytes
*                                    only used for ZUC-EIA/AES_CMAC algorithms.
*                                    default value 0 for the other algorithms.
* @param[out] ppHashDefsInfo         Hash Defs structure
*
* @return void
*
*****************************************************************************/
void LacSymQat_HashDefsLookupGet(CpaInstanceHandle instanceHandle,
                                 CpaCySymHashAlgorithm hashAlgorithm,
                                 lac_sym_qat_hash_defs_t **ppHashDefsInfo,
                                 Cpa32U authKeyLenInBytes,
                                 Cpa32U digestResultLenInBytes);

#endif /* LAC_SYM_QAT_HASH_DEFS_LOOKUP_P_H */
